import { NextRequest, NextResponse } from 'next/server';
import Database from 'better-sqlite3';
import path from 'path';

function getDb() {
    const dbPath = path.join(process.cwd(), 'comprehensive.db');
    return new Database(dbPath);
}

export async function GET(_: NextRequest, { params }: { params: Promise<{ id: string }> }) {
    const { id } = await params;
    const db = getDb();
    try {
        const row = db.prepare('SELECT * FROM tasks WHERE id = ?').get(id);
        if (!row) return NextResponse.json({ error: 'משימה לא נמצאה' }, { status: 404 });
        return NextResponse.json(row);
    } catch (e: any) {
        console.error('Get task error:', e);
        return NextResponse.json({ error: 'שגיאה בטעינת משימה: ' + e.message }, { status: 500 });
    } finally {
        db.close();
    }
}

export async function PUT(request: NextRequest, { params }: { params: Promise<{ id: string }> }) {
    const { id } = await params;
    const db = getDb();
    try {
        const body = await request.json();

        // וולידציה: אם יש domain_id וגם customer_id, לוודא שהם מתאימים
        if (body.domain_id && body.customer_id) {
            const domain = db.prepare('SELECT customer_id FROM domains WHERE id = ?').get(body.domain_id) as unknown;
            if (!domain) {
                return NextResponse.json(
                    { error: 'דומיין לא נמצא' },
                    { status: 400 }
                );
            }
            if (domain.customer_id !== body.customer_id) {
                return NextResponse.json(
                    { error: 'הדומיין לא שייך ללקוח שנבחר' },
                    { status: 400 }
                );
            }
        }

        // וולידציה: אם יש project_id וגם customer_id, לוודא שהם מתאימים
        if (body.project_id && body.customer_id) {
            const project = db.prepare('SELECT customer_id FROM projects WHERE id = ?').get(body.project_id) as unknown;
            if (!project) {
                return NextResponse.json(
                    { error: 'פרויקט לא נמצא' },
                    { status: 400 }
                );
            }
            if (project.customer_id !== body.customer_id) {
                return NextResponse.json(
                    { error: 'הפרויקט לא שייך ללקוח שנבחר' },
                    { status: 400 }
                );
            }
        }

        const fields = [
            'title', 'description', 'status', 'priority', 'category_id', 'customer_id', 'project_id',
            'assigned_to', 'due_date', 'completed_at'
        ];
        const setParts: string[] = [];
        const paramsObj: Record<string, unknown> = { id };
        for (const f of fields) {
            if (f in body) {
                setParts.push(`${f} = @${f}`);
                paramsObj[f] = (body as unknown)[f];
            }
        }
        if (!setParts.length) return NextResponse.json({ error: 'אין שדות לעדכון' }, { status: 400 });
        const sql = `UPDATE tasks SET ${setParts.join(', ')}, updated_at = CURRENT_TIMESTAMP WHERE id = @id`;
        const info = db.prepare(sql).run(paramsObj);
        if (info.changes === 0) return NextResponse.json({ error: 'משימה לא נמצאה' }, { status: 404 });
        const row = db.prepare('SELECT * FROM tasks WHERE id = ?').get(id);
        return NextResponse.json(row);
    } catch (e: any) {
        console.error('Update task error:', e);
        return NextResponse.json({
            error: e.message.includes('Domain does not belong')
                ? 'הדומיין לא שייך ללקוח שנבחר'
                : 'שגיאה בעדכון משימה: ' + e.message
        }, { status: 500 });
    } finally {
        db.close();
    }
}

export async function DELETE(_: NextRequest, { params }: { params: Promise<{ id: string }> }) {
    const { id } = await params;
    const db = getDb();
    try {
        const info = db.prepare('DELETE FROM tasks WHERE id = ?').run(id);
        if (info.changes === 0) return NextResponse.json({ error: 'משימה לא נמצאה' }, { status: 404 });
        return NextResponse.json({ ok: true, message: 'המשימה נמחקה בהצלחה' });
    } catch (e: any) {
        console.error('Delete task error:', e);
        return NextResponse.json({ error: 'שגיאה במחיקת משימה: ' + e.message }, { status: 500 });
    } finally {
        db.close();
    }
}


